#!/usr/bin/env bash
# mmp-stats.sh for cpuminer fork with ccminer/cgminer-style text API on 127.0.0.1:4048
# Outputs MMPOS-compatible JSON based on the miner "summary" and "threads" commands.

DEVICE_COUNT=$1
LOG_FILE=$2

# Miner API host/port
API_HOST="127.0.0.1"
API_PORT="4048"

# Fetch summary and threads data
SUMMARY_RAW=$(echo "summary" | nc -w 2 $API_HOST $API_PORT | tr -d '\0')
THREADS_RAW=$(echo "threads" | nc -w 2 $API_HOST $API_PORT | tr -d '\0')

# Fallback if API fails
if [[ -z "$SUMMARY_RAW" ]]; then
    echo '{"error": "Failed to fetch summary from miner API"}'
    exit 1
fi
if [[ -z "$THREADS_RAW" ]]; then
    echo '{"error": "Failed to fetch threads from miner API"}'
    exit 1
fi

# Parse summary line
# Example: NAME=cpuminer;VER=3.20;API=1.1;ALGO=sha256;CPUS=8;KHS=123.45;SOLV=10;ACC=20;REJ=0;ACCMN=5.000;DIFF=1.234000;TEMP=70.0;FAN=50;FREQ=3500;UPTIME=3600;TS=1620000000|
parse_field() {
    echo "$1" | grep -oP "(^|;)$2=\K[^;|]*"
}

MINER_NAME=$(parse_field "$SUMMARY_RAW" "NAME")
MINER_VERSION=$(parse_field "$SUMMARY_RAW" "VER")
UNITS="khs"
TOTAL_HASH=$(parse_field "$SUMMARY_RAW" "KHS")
ACCEPTED=$(parse_field "$SUMMARY_RAW" "ACC")
REJECTED=$(parse_field "$SUMMARY_RAW" "REJ")
TEMP=$(parse_field "$SUMMARY_RAW" "TEMP")
FREQ=$(parse_field "$SUMMARY_RAW" "FREQ")
UPTIME=$(parse_field "$SUMMARY_RAW" "UPTIME")

# Parse threads output for per-core stats
# Example: CPU=0;KHS=12.34|CPU=1;KHS=12.34|...
thread_hashes=()
IFS='|' read -ra THREADS <<< "$THREADS_RAW"
for t in "${THREADS[@]}"; do
    KH=$(echo "$t" | grep -oP "KHS=\K[0-9.]+")
    if [[ -n "$KH" ]]; then
        thread_hashes+=("$KH")
    fi
done

# Prepare JSON arrays
hash_json=$(printf '%s\n' "${thread_hashes[@]}" | jq -R . | jq -s .)
# For CPU miners, busid can be just CPU indices
uid_json=$(seq 0 $(( ${#thread_hashes[@]} - 1 )) | jq -R . | jq -s .)

# Shares: single CPU device, so array of one value
shares_json=$(jq -n --argjson accepted "[${ACCEPTED:-0}]" --argjson rejected "[${REJECTED:-0}]" --argjson invalid "[0]" \
    '{accepted: $accepted, rejected: $rejected, invalid: $invalid}')

# Final MMPOS-compatible JSON
jq -n \
    --arg miner_name "$MINER_NAME" \
    --arg miner_version "$MINER_VERSION" \
    --arg units "$UNITS" \
    --argjson uid "$uid_json" \
    --argjson hash "$hash_json" \
    --argjson shares "$shares_json" \
    '{
        uid: $uid,
        hash: $hash,
        units: $units,
        shares: $shares,
        miner_name: $miner_name,
        miner_version: $miner_version
    }'
