#!/usr/bin/env bash
# HiveOS miner_stat script for hoo-cpu

API_HOST="127.0.0.1"
API_PORT="4049"

# Query the miner API for summary and threads, remove null bytes
SUMMARY_RAW=$(echo "summary" | nc -w 2 $API_HOST $API_PORT | tr -d '\0')
THREADS_RAW=$(echo "threads" | nc -w 2 $API_HOST $API_PORT | tr -d '\0')

# Parse fields from summary output
parse_field() {
    echo "$1" | grep -oP "(^|;)$2=\K[^;|]*"
}

KHS=$(parse_field "$SUMMARY_RAW" "KHS")
ACC=$(parse_field "$SUMMARY_RAW" "ACC")
REJ=$(parse_field "$SUMMARY_RAW" "REJ")
UPTIME=$(parse_field "$SUMMARY_RAW" "UPTIME")
TEMP=$(parse_field "$SUMMARY_RAW" "TEMP")
FREQ=$(parse_field "$SUMMARY_RAW" "FREQ")

# Parse per-thread hashrate for HiveOS "hs" key
thread_hashes=()
IFS='|' read -ra THREADS <<< "$THREADS_RAW"
for t in "${THREADS[@]}"; do
    KH=$(echo "$t" | grep -oP "KHS=\K[0-9.]+")
    if [[ -n "$KH" ]]; then
        thread_hashes+=("$KH")
    fi
done

# JSON arrays for HiveOS
hash_json=$(printf '%s\n' "${thread_hashes[@]}" | jq -R . | jq -s .)
shares_json="[[${ACC:-0},${REJ:-0}]]"

# Compose stats JSON
stats=$(jq -nc \
  --argjson hs "$hash_json" \
  --argjson shares "$shares_json" \
  --arg uptime "${UPTIME:-0}" \
  --arg temp "${TEMP:-0}" \
  --arg freq "${FREQ:-0}" \
  '{
      hs: $hs,
      shares: $shares,
      uptime: ($uptime|tonumber),
      temp: [$temp|tonumber],
      freq: [$freq|tonumber]
  }'
)

# Export required HiveOS variables
khs="${KHS:-0}"
stats="$stats"
