#!/usr/bin/env bash
# HiveOS miner_stat script for hoo_gpu

API_HOST="127.0.0.1"
# ccminer/your API defaults to 4048 per api_server.cpp comment
API_PORT="${API_PORT:-4049}"

timeout_cmd() {
  # Busybox / GNU compatibility
  if command -v timeout >/dev/null 2>&1; then
    timeout 2 "$@"
  else
    "$@"
  fi
}

SUMMARY_RAW=$(echo "summary" | timeout_cmd nc -w 2 "$API_HOST" "$API_PORT" 2>/dev/null | tr -d '\0')
GPUS_RAW=$(echo "gpus" | timeout_cmd nc -w 2 "$API_HOST" "$API_PORT" 2>/dev/null | tr -d '\0')

parse_field() {
  echo "$1" | grep -oP "(^|;)$2=\K[^;|]*"
}

GKHS=$(parse_field "$SUMMARY_RAW" "GKHS")
GACC=$(parse_field "$SUMMARY_RAW" "GACC")
GREJ=$(parse_field "$SUMMARY_RAW" "GREJ")
UPTIME=$(parse_field "$SUMMARY_RAW" "UPTIME")

# Per-GPU parsing
gpu_hashes=()
gpu_temps=()
gpu_fans=()
gpu_powers=()
IFS='|' read -ra LINES <<< "$GPUS_RAW"
for line in "${LINES[@]}"; do
  [[ -z "$line" ]] && continue
  KH=$(echo "$line" | grep -oP "KHS=\K[0-9.]+")
  TEMP=$(echo "$line" | grep -oP "TEMP=\K[0-9]+")
  FAN=$(echo "$line" | grep -oP "FAN=\K[0-9]+")
  POW=$(echo "$line" | grep -oP "POW=\K[0-9.]+")
  if [[ -n "$KH" ]]; then gpu_hashes+=("$KH"); fi
  if [[ -n "$TEMP" ]]; then gpu_temps+=("$TEMP"); fi
  if [[ -n "$FAN" ]]; then gpu_fans+=("$FAN"); fi
  if [[ -n "$POW" ]]; then gpu_powers+=("$POW"); fi
done

# JSON arrays
hash_json=$(printf '%s\n' "${gpu_hashes[@]}" | jq -R . | jq -s .)
temp_json=$(printf '%s\n' "${gpu_temps[@]}" | jq -R . | jq -s .)
fan_json=$(printf '%s\n' "${gpu_fans[@]}" | jq -R . | jq -s .)
power_json=$(printf '%s\n' "${gpu_powers[@]}" | jq -R . | jq -s .)
algo='hoohash'

# HiveOS uses `ar` (accepted, rejected, stale) for the menu bar counters
A="${GACC:-0}"; R="${GREJ:-0}"
ar_json=$(printf '[%s,%s,0]' "${A:-0}" "${R:-0}")

# Keep `shares` for compatibility (some frontends still read it)
shares_json=$(printf '[[%s,%s]]' "${A:-0}" "${R:-0}")

stats=$(jq -nc \
  --arg total_khs "$khs" \
  --argjson hs "$hash_json" \
  --argjson ar "$ar_json" \
  --arg uptime "${UPTIME:-0}" \
  --argjson temp "$temp_json" \
  --argjson fan "$fan_json" \
  --arg algo "$algo" \
  --argjson power "$power_json" \
  --argjson shares "$shares_json" \
  '{
      hs: $hs,
      ar: $ar,
      shares: $shares,
      uptime: ($uptime|tonumber),
      temp: $temp,
      fan: $fan,
      algo: $algo,
      power: $power
  }'
)

khs="${GKHS:-0}"
echo "khs=$khs"
echo "stats=$stats"


