#!/bin/bash
# Runs Hoominer, Hoo-CPU, and Hoo-GPU in parallel and fuses their stats (khs, hs, temp, fan, etc.) into a single JSON object

. h-manifest.conf

# Check for required tools
check_tool() {
    if ! command -v "$1" >/dev/null 2>&1; then
        echo "Error: $1 is required but not installed" >&2
        exit 1
    fi
}

# Check for required tools
check_tool "jq"
check_tool "bc"
check_tool "nc"

# Read config file if it exists
if [ -f "$CUSTOM_CONFIG_FILENAME" ]; then
    CONFIG_CONTENTS=$(< "$CUSTOM_CONFIG_FILENAME")
else
    echo "Warning: Config file $CUSTOM_CONFIG_FILENAME not found" >&2
    CONFIG_CONTENTS=""
fi

# -------- Hoominer --------
if [[ "$CONFIG_CONTENTS" == *"--hoominer"* ]]; then
    # Fetch stats from Hoominer API
    stats_raw=$(curl -s http://127.0.0.1:8042/gpu)
    if [ $? -ne 0 ]; then
        echo "Failed to fetch stats from Hoominer API"
        exit 1
    fi

    # Check if stats_raw is empty or invalid JSON
    if [ -z "$stats_raw" ] || ! echo "$stats_raw" | jq . >/dev/null 2>&1; then
        echo "Invalid or empty API response"
        exit 1
    fi

    # Parse JSON using jq
    khs=$(echo "$stats_raw" | jq '[.hash[] / 1000] | add // 0')
    hs=$(echo "$stats_raw" | jq '[.hash[] / 1000] // []')
    busid=$(echo "$stats_raw" | jq '[.busid[] | if . == "cpu" then 0 else . end] // []')
    air=$(echo "$stats_raw" | jq '.air // 0')
    accepted=$(echo "$stats_raw" | jq '.shares.accepted | add // 0')
    rejected=$(echo "$stats_raw" | jq '.shares.rejected | add // 0')
    hs_units="khs"
    ver=$(echo "$stats_raw" | jq -r '.miner_version // "unknown"')
    algo='hoohash'

    # Calculate uptime
    pid=$(pgrep -f hoominer | head -n1)
    if [ -n "$pid" ]; then
        # Get uptime in seconds using ps
        uptime_seconds=$(ps -p "$pid" -o etimes= | tr -d '[:space:]')
        uptime=${uptime_seconds:-0}
    else
        uptime="0"
    fi

    hs_length=$(echo "$hs" | jq 'length')
    temp_length=$(echo "$temp" | jq 'length')
    fan_length=$(echo "$fan" | jq 'length')

    # Remove first element from temp and fan if hs array is shorter
    if [ "$hs_length" -lt "$temp_length" ]; then
        temp=$(echo "$temp" | jq '.[1:] // .')
    fi
    if [ "$hs_length" -lt "$fan_length" ]; then
        fan=$(echo "$fan" | jq '.[1:] // .')
    fi

    # Format stats for HiveOS
    stats=$(jq -n \
        --arg total_khs "$khs" \
        --argjson hs "$hs" \
        --arg hs_units "$hs_units" \
        --argjson temp "$temp" \
        --argjson fan "$fan" \
        --argjson bus_numbers "$busid" \
        --argjson accepted "$accepted" \
        --argjson rejected "$rejected" \
        --arg uptime "$uptime" \
        --arg ver "$ver" \
        --arg algo "$algo" \
        '{
            total_khs: $total_khs,
            hs: $hs,
            hs_units: $hs_units,
            temp: $temp,
            fan: $fan,
            bus_numbers: $bus_numbers,
            ar: [$accepted, $rejected],
            algo: $algo,
            uptime: $uptime,
            ver: $ver
        }')

    echo "$stats"
elif [[ "$CONFIG_CONTENTS" == *"--hoo_cpu"* ]]; then
    # HiveOS miner_stat script for hoo-cpu
    API_HOST="127.0.0.1"
    API_PORT="4048"

    # Query the miner API for summary and threads, remove null bytes
    SUMMARY_RAW=$(echo "summary" | nc -w 2 $API_HOST $API_PORT | tr -d '\0')
    THREADS_RAW=$(echo "threads" | nc -w 2 $API_HOST $API_PORT | tr -d '\0')

    # Parse fields from summary output
    parse_field() {
        echo "$1" | grep -oP "(^|;)$2=\K[^;|]*"
    }

    KHS=$(parse_field "$SUMMARY_RAW" "KHS")
    ACC=$(parse_field "$SUMMARY_RAW" "ACC")
    REJ=$(parse_field "$SUMMARY_RAW" "REJ")
    UPTIME=$(parse_field "$SUMMARY_RAW" "UPTIME")
    TEMP=$(parse_field "$SUMMARY_RAW" "TEMP")
    FREQ=$(parse_field "$SUMMARY_RAW" "FREQ")

    # Parse per-thread hashrate for HiveOS "hs" key
    thread_hashes=()
    IFS='|' read -ra THREADS <<< "$THREADS_RAW"
    for t in "${THREADS[@]}"; do
        KH=$(echo "$t" | grep -oP "KHS=\K[0-9.]+")
        if [[ -n "$KH" ]]; then
            thread_hashes+=("$KH")
        fi
    done

    # JSON arrays for HiveOS
    hash_json=$(printf '%s\n' "${thread_hashes[@]}" | jq -R . | jq -s .)
    shares_json="[[${ACC:-0},${REJ:-0}]]"

    # Compose stats JSON
    stats=$(jq -nc \
    --argjson hs "$hash_json" \
    --argjson shares "$shares_json" \
    --arg uptime "${UPTIME:-0}" \
    --arg temp "${TEMP:-0}" \
    --arg freq "${FREQ:-0}" \
    '{
        hs: $hs,
        shares: $shares,
        uptime: ($uptime|tonumber),
        temp: [$temp|tonumber],
        freq: [$freq|tonumber]
    }'
    )

    # Export required HiveOS variables
    khs="${KHS:-0}"
    stats="$stats"

elif [[ "$CONFIG_CONTENTS" == *"--hoo_gpu"* ]]; then
    API_HOST="127.0.0.1"
    # ccminer/your API defaults to 4048 per api_server.cpp comment
    API_PORT="${API_PORT:-4049}"

    timeout_cmd() {
    # Busybox / GNU compatibility
    if command -v timeout >/dev/null 2>&1; then
        timeout 2 "$@"
    else
        "$@"
    fi
    }

    SUMMARY_RAW=$(echo "summary" | timeout_cmd nc -w 2 "$API_HOST" "$API_PORT" 2>/dev/null | tr -d '\0')
    GPUS_RAW=$(echo "gpus" | timeout_cmd nc -w 2 "$API_HOST" "$API_PORT" 2>/dev/null | tr -d '\0')

    parse_field() {
    echo "$1" | grep -oP "(^|;)$2=\K[^;|]*"
    }

    GKHS=$(parse_field "$SUMMARY_RAW" "GKHS")
    GACC=$(parse_field "$SUMMARY_RAW" "GACC")
    GREJ=$(parse_field "$SUMMARY_RAW" "GREJ")
    UPTIME=$(parse_field "$SUMMARY_RAW" "UPTIME")

    # Per-GPU parsing
    gpu_hashes=()
    gpu_temps=()
    gpu_fans=()
    gpu_powers=()
    IFS='|' read -ra LINES <<< "$GPUS_RAW"
    for line in "${LINES[@]}"; do
    [[ -z "$line" ]] && continue
    KH=$(echo "$line" | grep -oP "KHS=\K[0-9.]+")
    TEMP=$(echo "$line" | grep -oP "TEMP=\K[0-9]+")
    FAN=$(echo "$line" | grep -oP "FAN=\K[0-9]+")
    POW=$(echo "$line" | grep -oP "POW=\K[0-9.]+")
    if [[ -n "$KH" ]]; then gpu_hashes+=("$KH"); fi
    if [[ -n "$TEMP" ]]; then gpu_temps+=("$TEMP"); fi
    if [[ -n "$FAN" ]]; then gpu_fans+=("$FAN"); fi
    if [[ -n "$POW" ]]; then gpu_powers+=("$POW"); fi
    done

    # JSON arrays
    hash_json=$(printf '%s\n' "${gpu_hashes[@]}" | jq -R . | jq -s .)
    temp_json=$(printf '%s\n' "${gpu_temps[@]}" | jq -R . | jq -s .)
    fan_json=$(printf '%s\n' "${gpu_fans[@]}" | jq -R . | jq -s .)
    power_json=$(printf '%s\n' "${gpu_powers[@]}" | jq -R . | jq -s .)
    algo='hoohash'

    # HiveOS uses `ar` (accepted, rejected, stale) for the menu bar counters
    A="${GACC:-0}"; R="${GREJ:-0}"
    ar_json=$(printf '[%s,%s,0]' "${A:-0}" "${R:-0}")

    # Keep `shares` for compatibility (some frontends still read it)
    shares_json=$(printf '[[%s,%s]]' "${A:-0}" "${R:-0}")

    stats=$(jq -nc \
    --arg total_khs "$khs" \
    --argjson hs "$hash_json" \
    --argjson ar "$ar_json" \
    --arg uptime "${UPTIME:-0}" \
    --argjson temp "$temp_json" \
    --argjson fan "$fan_json" \
    --arg algo "$algo" \
    --argjson power "$power_json" \
    --argjson shares "$shares_json" \
    '{
        hs: $hs,
        ar: $ar,
        shares: $shares,
        uptime: ($uptime|tonumber),
        temp: $temp,
        fan: $fan,
        algo: $algo,
        power: $power
    }'
    )

    khs="${GKHS:-0}"
    echo "khs=$khs"
    echo "stats=$stats"
fi
